---
title: Pipes
localeTitle: трубы
---
# трубы

#### мотивация

Преобразование выходных данных. Они обеспечивают, чтобы данные были в желаемом формате к моменту загрузки на экран пользователя. Обычно данные преобразуются за кулисами. С помощью труб преобразующие данные могут иметь место в шаблоне HTML. Трубы напрямую преобразуют данные шаблона.

Трубы выглядят красиво и удобны. Они помогают сохранить класс компонентов базовых преобразований. Технически говоря, трубы инкапсулируют логику преобразования данных.

#### Выходная трансформация

Как упоминалось в предыдущем разделе, трубы преобразуют данные в линию. Синтаксис труб коррелирует с командами оболочки. Во многих сценариях выход одной части команды _передается в_ виде вывода в следующую часть как входной. Эта же тенденция характеризует Угловые трубы.

В Angular существует множество способов взаимодействия с данными в HTML-шаблоне. Трубы могут применяться везде, где данные обрабатываются в HTML-шаблоне. Они могут встречаться в рамках микросинтетической логики и внутренних интерполяций innerHTML. Трубы учитывают все преобразования без добавления в класс компонентов.

Трубы также соединены с _цепью_ . Вы можете интегрировать трубы один за другим, чтобы выполнять все более сложные преобразования. Как специализированные трансформаторы данных, трубы едва ли тривиальны.

#### Случаи применения

Угловой поставляется с расфасованным базовым набором труб. Работа с несколькими из них будет развивать интуицию, чтобы справиться с остальными. В следующем списке представлены два примера.

*   AsyncPipe
    
*   DatePipe
    

Эти два выполняют простые задачи. Их простота очень полезна.

##### AsyncPipe

В этих разделах требуется полное понимание обещаний или наблюдений в полной мере. AsyncPipe работает на любом из двух. AsyncPipe извлекает данные из Promises / Observables в качестве вывода для следующего.

В случае Obervables, AsyncPipe автоматически подписывается на источник данных. Независимо от того, откуда поступают данные, AsyncPipe подписывается на наблюдаемый источник. `async` - это синтаксическое имя AsyncPipe, как показано ниже.

```html

<ul *ngFor=“let potato of (potatoSack$ | async); let i=index”> 
  <li>Potatoe {{i + 1}}: {{potato}}</li> 
 </ul> 
```

В примере, `potatoSack$` является наблюдаемым в ожидании загрузки картофеля. Как только картофель прибывает, синхронно или асинхронно, AsyncPipe получает их как _итерируемый_ массив. Затем элемент списка заполняется картофелем.

##### DatePipe

Строки даты форматирования принимают справедливый бит взлома с объектом JavaScript `Date` . DatePipe обеспечивает мощный способ форматирования дат, если данный ввод является допустимым временным форматом.

```typescript
// example.component.ts 
 
 @Component({ 
  templateUrl: './example.component.html' 
 }) 
 export class ExampleComponent { 
  timestamp:string = '2018-05-24T19:38:11.103Z'; 
 } 
```

```html

<!-- example.component.html --> 
 
 <div>Current Time: {{timestamp | date:'short'}}</div> 
```

Формат указанной `timestamp` - [ISO 8601 1](https://en.wikipedia.org/wiki/ISO_8601) - не самый простой для чтения. DatePipe ( `date` ) преобразует формат даты ISO в более обычный формат `mm/dd/yy, hh:mm AM|PM` . Есть много других вариантов форматирования. Все эти варианты указаны в [официальной документации](https://angular.io/api/common/DatePipe) .

#### Создание труб

В то время как у Angular есть только определенное количество труб, декоратор `@Pipe` позволяет разработчикам создавать свои собственные. Процесс начинается с `ng generate pipe [name-of-pipe]` , заменяя `[name-of-pipe]` предпочтительным именем файла. Это команда [Angular CLI](https://cli.angular.io) . Это дает следующее.

```typescript
import { Pipe, PipeTransform } from '@angular/core'; 
 
 @Pipe({ 
  name: 'example' 
 }) 
 export class ExamplePipe implements PipeTransform { 
  transform(value: any, args?: any): any { 
      return null; 
  } 
 } 
```

Этот шаблон трубы упрощает создание пользовательских труб. Декоратор `@Pipe` сообщает, что «Угловой» класс является трубой. Значение `name: 'example'` , в данном случае являющееся `example` , является значением, которое Angular распознает при сканировании HTML-шаблона для пользовательских каналов.

На классную логику. Реализация `PipeTransform` предоставляет инструкции для функции `transform` . Эта функция имеет особое значение в контексте декоратора `@Pipe` . По умолчанию он получает два параметра.

`value: any` - это результат, который получает труба. Подумайте о `<div>{{ someValue | example }}</div>` . Значение someValue передается в значение функции `transform` `value: any` параметр. Это та же функция `transform` которая определена в классе ExamplePipe.

`args?: any` - любой аргумент, который канал получает дополнительно. Подумайте о `<div>{{ someValue | example:[some-argument] }}</div>` . `[some-argument]` можно заменить любым значением. Это значение передается `args?: any` функции `transform` `args?: any` параметр. То есть, функция `transform` определена в классе ExamplePipe.

Независимо от того, что возвращает функция ( `return null;` ), становится результатом работы трубы. Взгляните на следующий пример, чтобы увидеть полный пример ExamplePipe. В зависимости от переменной, получаемой трубой, она либо верхние или нижние регистры вводят в качестве нового выхода. Недействительный или несуществующий аргумент заставит канал возвращать тот же ввод, что и вывод.

```typescript
// example.pipe.ts 
 
 @Pipe({ 
  name: 'example' 
 }) 
 export class ExamplePipe implements PipeTransform { 
  transform(value:string, args?:string): any { 
    switch(args || null) { 
      case 'uppercase': 
        return value.toUpperCase(); 
      case 'lowercase': 
        return value.toLowerCase(); 
      default: 
        return value; 
    } 
  } 
 } 
```

```typescript
// app.component.ts 
 
 @Component({ 
  templateUrl: 'app.component.html' 
 }) 
 export class AppComponent { 
  someValue:string = "HeLlO WoRlD!"; 
 } 
```

```html

<!-- app.component.html --> 
 
 <!-- Outputs “HeLlO WoRlD!” --> 
 <h6>{{ someValue | example }}</h6> 
 
 <!-- Outputs “HELLO WORLD!” --> 
 <h6>{{ someValue | example:'uppercase' }}</h6> 
 
 <!-- Outputs “hello world!” --> 
 <h6>{{ someValue | example:'lowercase' }}</h6> 
```

Понимание приведенного выше примера означает, что вы понимаете угловые трубы. Остается обсудить еще одну тему.

#### Чистые и нечистые трубы

Все, что вы видели до сих пор, было _чистой_ трубой. `pure: true` по умолчанию `@Pipe` в метаданных декоратора `@Pipe` . Разница между двумя составляет обнаружение изменений Углового.

Чистые каналы обновляются автоматически, когда изменяется значение его производного ввода. Труба будет повторно выполняться для получения нового выхода после обнаружения изменения входного значения. Обнаруживаемые изменения определяются циклом обнаружения изменений угла.

Непрочные трубы обновляются автоматически, когда происходит цикл обнаружения изменений. Обнаружение изменений углов происходит довольно часто. Он сигнализирует, произошли ли изменения в данных элемента компонента. Если это так, шаблон HTML обновляется с обновленными данными. В противном случае ничего не произойдет.

В случае нечистого трубопровода он будет обновляться независимо от того, есть ли обнаруженное изменение или нет. Нечистая труба обновляется всякий раз, когда меняются контуры обнаружения изменений. Нечистые трубы обычно потребляют много ресурсов и обычно плохо информированы. Тем не менее, они действуют больше как побег. Если вам когда-нибудь понадобится чувствительная к обнаружению труба, переключите `pure: false` в `@Pipe` декоратора `@Pipe` .

#### Вывод

Это покрывает трубы. Трубы обладают большим потенциалом, выходящим за рамки этой статьи. Трубы вносят сжатые преобразования данных в HTML-код вашего компонента. Они являются хорошей практикой в ​​ситуациях, когда данные должны претерпевать небольшие изменения.

## источники

1.  [Сообщество Wiki. _ISO 8601_ . Wikipedia. Доступ к 27 мая 2018 года](https://en.wikipedia.org/wiki/ISO_8601)

## Ресурсы

*   [Угловая документация](https://angular.io/guide/pipes)
*   [Угловой репозиторий GitHub](https://github.com/angular/angular)
*   [Список труб, предварительно упакованных с угловым](https://angular.io/api?query=pipe)
*   [Угловая CLI](https://cli.angular.io)