---
title: Casting
localeTitle: Кастинг
---
## Кастинг

Литье - это специальный оператор, который заставляет один тип данных преобразовываться в другой

Кастинг в C ++ несколько отличается от C C. C ++ использует различные функции кастингов.

### static\_cast

Static cast используется для неявных преобразований между примитивами и перегрузками типов.

### const\_cast

Const cast может использоваться для изгнания константы. Это полезно, когда есть желание изменить постоянное значение. Это следует использовать экономно, вместо этого следует рассмотреть возможность создания параметров / функций не const в тех случаях, когда используется const-cast.

Const cast также может привести к неопределенному поведению. Единственное приложение const cast должно всегда состоять в том, чтобы удалить константу из значения, которое было передано функции и помечено как const. Если значение истинно const, то есть оно помечено как const во время компиляции и назначено значение, const cast и мутация переменной приведет к неопределенному поведению.
```
const int y = 10;             // y is set to 10. 
 const_cast<int &>(y) = 20;    // undefined behaviour. 
```

### dynamic\_cast

Динамический кастинг используется для создания объекта внутри его иерархии классов (родительскому, от родительского и к родному брату). Динамический бросок может быть вызван только для полиморфных классов. Таким образом, исходный класс в этом случае `MyClass` должен иметь виртуальный член, который присутствует в форме виртуального деструктора.

Если динамическое сканирование завершится неудачно, оно вернет `nullptr` . Динамический листинг может быть полезен при определении типов объектов во время выполнения. Однако следует отметить, что динамический кастинг не является бесплатным, и в некоторых случаях другие методы могут оказаться более эффективными при определении типа класса во время выполнения.

### reinterpret\_cast

Reinterpret cast - это, пожалуй, самый опасный из всех C ++-приведений, но при правильном использовании он может быть идеальным. Реинтерпрет не влияет на производительность, поскольку он не выполняет никаких преобразований. Он просто инструктирует компилятор обрабатывать литой объект, как если бы он был запрошенным типом. Это также может привести к проблемам выравнивания, поэтому его следует использовать экономно и только тогда, когда известны побочные эффекты и учитываются.

#### Заметка о стилях C-стиля

C ++ поддерживает использование стилей C-стиля, хотя они не рекомендуются. Использование стилей C-стиля даст указание компилятору сначала выполнить статическую _актерскую съемку, если статический_ приведение не выполняется, reinterpret\_cast используется на своем месте. По этой причине приведение в стиле C может привести к непредсказуемым результатам и вызвать неожиданные проблемы.

## Примеры

```cpp
#include <iostream> 
 
 class MyClass { 
 public: 
    virtual ~MyClass() = default; 
 
    void greet() { 
        std::cout << "Hello World!" << std::endl; 
    } 
 }; 
 
 class MyClassChild : public MyClass { 
 }; 
 
 void reinterpretCastTest(void *objectPtr) { 
    // Let's assume we know objectPtr is of type MyClass * 
    auto myClassObj = reinterpret_cast<MyClassChild *>(objectPtr); 
    myClassObj->greet(); 
 } 
 
 void constCastTest(const MyClassChild &myClassChild) { 
    auto nonConst = const_cast<MyClassChild &>(myClassChild); 
    nonConst.greet(); 
 } 
 
 void dynamicCastTest(MyClass *myClass) { 
    auto *child = dynamic_cast<MyClassChild *>(myClass); 
    child->greet(); 
 } 
 
 void staticCastTest(float floatVal) { 
    // Convert the float into an int. 
    auto intVal = static_cast<int>(floatVal); 
    std::cout << intVal << std::endl; 
 } 
 
 int main() { 
    MyClassChild myClass; 
    reinterpretCastTest(&myClass); 
    constCastTest(myClass); 
    dynamicCastTest(&myClass); 
    staticCastTest(10.5); 
 
    return 0; 
 } 

```