---
title: Separate Build Image
localeTitle: Отдельное изображение сборки
---
## обзор

Создание облегченных изображений докеров является ключом к быстрому развитию / развертыванию. Для скомпилированного кода построение двоичного файла внутри контейнера докеров имеет преимущество в виде повторяемого и стандартизованного процесса сборки. Однако это может создать очень большие изображения, которые могут стать проблемой на линии.

## Наш код

В этом примере мы будем использовать простой сценарий веб-сервера в [Go](https://golang.org/) . Следующий код - просто простой веб-сервер приветствия, прослушивающий порт `8080` .

```go
package main 
 
 import ( 
    "fmt" 
    "log" 
    "net/http" 
 ) 
 
 func handler(w http.ResponseWriter, r *http.Request) { 
    fmt.Fprint(w, "Hello world!") 
 } 
 
 func main() { 
    http.HandleFunc("/", handler) 
    log.Fatal(http.ListenAndServe(":8080", nil)) 
 } 
```

### Dockerfile

Файл Docker для этого кода может выглядеть примерно так:
```
FROM golang:1.11 
 
 ADD . /app 
 
 WORKDIR /app 
 
 RUN go build -o /myserver . 
 
 EXPOSE 8080 
 
 CMD [ "/myserver" ] 
```

Построение этого изображения приводит к изображению размером 783 МБ !! С изображением такого размера для простого приложения легко понять, как это может замедлить работу при развертывании.

## Лучшее решение

Лучшим решением было бы использовать отдельный образ сборки для создания двоичного файла, а затем скопировать его в окончательное изображение. Поскольку Go генерирует автономный двоичный файл, мы можем использовать изображение докеры с `scratch` в качестве базы, которая примерно такая же маленькая, как и она!

### Dockerfile

Следующий файл Dockerfile сначала построит двоичный файл внутри изображения golang, а затем построит новое изображение с нуля, скопировав двоичный файл с первого изображения на второй.
```
FROM golang:1.11 as build 
 
 ADD . /app 
 
 WORKDIR /app 
 
 RUN go build -o /myserver . 
 
 
 FROM scratch 
 
 COPY --from=build /myserver /myserver 
 
 EXPOSE 8080 
 
 CMD [ "myserver" ] 
```

Создание из этого файла dockerfile приводит к окончательному размеру изображения всего 6.55MB! Это более чем в **100 раз меньше** нашей первой попытки, сделав ее в 100 раз быстрее, чтобы вытащить изображение из реестра!

### Бонусная выгода

Мы не только имеем крошечный образ докеры для нашего приложения, но и беспокоимся о безопасности нашего приложения, поскольку в нем нет другого программного обеспечения.