---
title: Function Composition
localeTitle: Состав функции
---
## Состав функции

Состав функции - это поточечное применение одной функции к результату другого. Разработчики делают это в ручном режиме каждый день, когда гнездо функционирует:

```javascript
compose = (fn1, fn2) => value => fn2(fn1(value)) 
```

Но это трудно читать. Существует лучший способ использования композиции функций. Вместо того, чтобы читать их изнутри:

```javascript
add2AndSquare = (n) => square(add2(n)) 
```

Мы можем использовать функцию более высокого порядка, чтобы упорядочить их упорядоченно.

```javascript
add2AndSquare = compose( add2, square) 
```

Простая реализация компоновки будет заключаться в следующем:

```javascript
compose = (f1, f2) => value => f2( f1(value) ); 
```

Чтобы получить еще большую гибкость, мы можем использовать функцию reduceRight:

```javascript
compose = (...fns) => (initialVal) => fns.reduceRight((val, fn) => fn(val), initialVal); 
```

Чтение композиции слева направо позволяет четко связывать функции более высокого порядка. В реальных примерах мира добавляются аутентификации, журналирование и свойства контекста. Это метод, который позволяет повторно использовать на самом высоком уровне. Вот несколько примеров использования:

```javascript
// example 
 const add2        = (n) => n + 2; 
 const times2      = (n) => n * 2; 
 const times2add2  = compose(add2, times2); 
 const add6        = compose(add2, add2, add2); 
 
 times2add2(2);  // 6 
 add2tiems2(2);  // 8 
 add6(2);        // 8 
```

Вы можете подумать, что это расширенное функциональное программирование, и это не относится к программированию интерфейса. Но это также полезно в приложениях с одной страницей. Например, вы можете добавить поведение к компоненту React, используя компоненты более высокого порядка:

```javascript
function logProps(InputComponent) { 
  InputComponent.prototype.componentWillReceiveProps = function(nextProps) { 
    console.log('Current props: ', this.props); 
    console.log('Next props: ', nextProps); 
  }; 
  return InputComponent; 
 } 
 
 // EnhancedComponent will log whenever props are received 
 const EnhancedComponent = logProps(InputComponent); 
```

В заключение функциональный состав обеспечивает возможность повторного использования функциональности на очень высоком уровне. Если функции хорошо структурированы, это позволяет разработчикам создавать новое поведение на основе существующего поведения.

Это также повышает читаемость реализаций. Вместо функций вложенности вы можете очищать функции цепочки и создавать функции более высокого порядка со значимыми именами.