---
title: REST APIs with Falcon
localeTitle: API REST с Falcon
---
## Введение

API RESTful являются основным компонентом любого хорошо продуманного стека, и у Python есть некоторые блестящие рамки для быстрого составления API. Одна из этих фреймов называется [Falcon](https://falconframework.org) - и это здорово! По существу микрокарта, она поставляется с большим количеством преимуществ:

1.  Это быстро. Действительно быстро. Проверьте тесты [здесь](https://falconframework.org/#sectionBenchmarks) .
    
2.  Ресурсы HTTP определяются как классы, при этом методы класса используются для разных операций REST на этих ресурсах. Это помогает поддерживать чистую кодовую базу.
    
3.  Это довольно расширяемо - ознакомьтесь с [этим разделом](https://github.com/falconry/falcon/wiki/Complementary-Packages) на своей вики, чтобы почувствовать это.
    
4.  Он основан на WSGI - стандарте Pythonic для веб-приложений, поэтому он работает с Python 2.6, 2.7 и 3.3+. И если вам нужно больше производительности, запустите его с помощью PyPy!
    

## Начиная

Во-первых, давайте подготовим нашу среду. Лично, всегда здорово работать в виртуальных средах - вы можете использовать `virtualenv` , `virtualenvwrapper` или `venv` . Затем, установите Falcon, используя `pip` : `pip install falcon` .

Мы собираемся разработать небольшой образец API, который делает для нас основные манипуляции с часовыми поясами. Он отображает текущее время в UTC, а также соответствующее время эпохи. Для этого мы возьмем отличную библиотеку со `arrow` : `pip install arrow` .

Вы можете найти готовый образец на [странице https://github.com/rudimk/freecodecamp-guides-rest-api-falcon](https://github.com/rudimk/freecodecamp-guides-rest-api-falcon) .

## Ресурсы

Подумайте о ресурсе как об объекте, которым должен манипулировать ваш API. В нашем случае лучшим ресурсом будет `Timestamp` . Наша маршрутизация обычно будет примерно такой:
```
GET /timestamp 
```

Здесь `GET` - это HTTP-глагол, используемый для вызова этой конечной точки, а `/timestamp` - это сам URL. Теперь, когда мы получили это немного, давайте создадим модуль!

`$ touch timestamp.py`

Время для импорта библиотеки Falcon:

```python
import json 
 
 import falcon 
 
 import arrow 
```

Обратите внимание, что мы также импортируем пакет `json` и библиотеку `arrow` . Теперь давайте определим класс для нашего ресурса:

```python
class Timestamp(object): 
 
    def on_get(self, req, resp): 
        payload = {} 
        payload['utc'] = arrow.utcnow().format('YYYY-MM-DD HH:mm:SS') 
        payload['unix'] = arrow.utcnow().timestamp 
 
        resp.body = json.dumps(payload) 
        resp.status = falcon.HTTP_200 
```

Давайте рассмотрим этот фрагмент. Мы определили класс `Timestamp` и определили метод класса, называемый `on_get` - эта функция сообщает Falcon, что когда `GET` запрос выдается конечной точке для этого ресурса, запустите функцию `on_get` и предоставите объекты запроса и ответа в качестве параметров. После этого плавное плавание - мы создаем пустой словарь, заполняем его текущими отметками времени UTC и UNIX, конвертируем их в JSON и присоединяем к объекту ответа.

Довольно просто, не так ли? Но, к сожалению, это еще не все. Теперь нам нужно создать сервер Falcon и подключить класс ресурсов, который мы только что определили, к фактической конечной точке.

`$ touch app.py`

Теперь добавьте следующий код:

```python
import falcon 
 
 from timestamp import Timestamp 
 
 api = application = falcon.API() 
 
 timestamp = Timestamp() 
 
 api.add_route('/timestamp', timestamp) 
```

Здесь мы определили API Falcon и инициализировали экземпляр класса ресурсов, который мы создали ранее. Затем мы подключили конечную точку `/timestamp` с экземпляром класса - и теперь мы готовы к работе! Чтобы проверить этот API, установите `gunicorn` ( `pip install gunicorn` ) и запустите `gunicorn app` . Используйте Postman или простую `cURL` чтобы проверить это:
```
$ curl http://localhost:8000/timestamp 
 {"utc": "2017-10-20 06:03:14", "unix": 1508479437} 
```

И это все!

## Перемещение

Как только вы приобретете Falcon, создание мощных API RESTful, которые взаимодействуют с базами данных или очередями обмена сообщениями, очень просто. [Просмотрите документы Falcon](https://falcon.readthedocs.io/en/stable/index.html) , а также PyPI для интересных модулей Falcon, которые продолжают появляться.