"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const typescript_estree_1 = require("@typescript-eslint/typescript-estree");
const util = __importStar(require("../util"));
exports.default = util.createRule({
    name: 'adjacent-overload-signatures',
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Require that member overloads be consecutive',
            category: 'Best Practices',
            tslintName: 'adjacent-overload-signatures',
            recommended: 'error',
        },
        schema: [],
        messages: {
            adjacentSignature: "All '{{name}}' signatures should be adjacent.",
        },
    },
    defaultOptions: [],
    create(context) {
        const sourceCode = context.getSourceCode();
        /**
         * Gets the name of the member being processed.
         * @param member the member being processed.
         * @returns the name of the member or null if it's a member not relevant to the rule.
         */
        function getMemberName(member) {
            if (!member) {
                return null;
            }
            switch (member.type) {
                case typescript_estree_1.AST_NODE_TYPES.ExportDefaultDeclaration:
                case typescript_estree_1.AST_NODE_TYPES.ExportNamedDeclaration: {
                    // export statements (e.g. export { a };)
                    // have no declarations, so ignore them
                    if (!member.declaration) {
                        return null;
                    }
                    return getMemberName(member.declaration);
                }
                case typescript_estree_1.AST_NODE_TYPES.TSDeclareFunction:
                case typescript_estree_1.AST_NODE_TYPES.FunctionDeclaration:
                    return member.id && member.id.name;
                case typescript_estree_1.AST_NODE_TYPES.TSMethodSignature:
                    return util.getNameFromPropertyName(member.key);
                case typescript_estree_1.AST_NODE_TYPES.TSCallSignatureDeclaration:
                    return 'call';
                case typescript_estree_1.AST_NODE_TYPES.TSConstructSignatureDeclaration:
                    return 'new';
                case typescript_estree_1.AST_NODE_TYPES.MethodDefinition:
                    return util.getNameFromClassMember(member, sourceCode);
            }
            return null;
        }
        function isSameMethod(method1, method2) {
            return (!!method2 &&
                method1.name === method2.name &&
                method1.static === method2.static);
        }
        function getMembers(node) {
            switch (node.type) {
                case typescript_estree_1.AST_NODE_TYPES.ClassBody:
                case typescript_estree_1.AST_NODE_TYPES.Program:
                case typescript_estree_1.AST_NODE_TYPES.TSModuleBlock:
                case typescript_estree_1.AST_NODE_TYPES.TSInterfaceBody:
                    return node.body;
                case typescript_estree_1.AST_NODE_TYPES.TSTypeLiteral:
                    return node.members;
            }
            return [];
        }
        /**
         * Check the body for overload methods.
         * @param {ASTNode} node the body to be inspected.
         */
        function checkBodyForOverloadMethods(node) {
            const members = getMembers(node);
            if (members) {
                let lastMethod = null;
                const seenMethods = [];
                members.forEach(member => {
                    const name = getMemberName(member);
                    if (name === null) {
                        lastMethod = null;
                        return;
                    }
                    const method = {
                        name,
                        static: 'static' in member && !!member.static,
                    };
                    const index = seenMethods.findIndex(seenMethod => isSameMethod(method, seenMethod));
                    if (index > -1 && !isSameMethod(method, lastMethod)) {
                        context.report({
                            node: member,
                            messageId: 'adjacentSignature',
                            data: {
                                name: (method.static ? 'static ' : '') + method.name,
                            },
                        });
                    }
                    else if (index === -1) {
                        seenMethods.push(method);
                    }
                    lastMethod = method;
                });
            }
        }
        return {
            ClassBody: checkBodyForOverloadMethods,
            Program: checkBodyForOverloadMethods,
            TSModuleBlock: checkBodyForOverloadMethods,
            TSTypeLiteral: checkBodyForOverloadMethods,
            TSInterfaceBody: checkBodyForOverloadMethods,
        };
    },
});
//# sourceMappingURL=adjacent-overload-signatures.js.map