"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const util = __importStar(require("../util"));
exports.default = util.createRule({
    name: 'explicit-member-accessibility',
    meta: {
        type: 'problem',
        docs: {
            description: 'Require explicit accessibility modifiers on class properties and methods',
            tslintRuleName: 'member-access',
            category: 'Best Practices',
            recommended: 'error',
        },
        messages: {
            missingAccessibility: 'Missing accessibility modifier on {{type}} {{name}}.',
        },
        schema: [],
    },
    defaultOptions: [],
    create(context) {
        const sourceCode = context.getSourceCode();
        /**
         * Checks if a method declaration has an accessibility modifier.
         * @param methodDefinition The node representing a MethodDefinition.
         */
        function checkMethodAccessibilityModifier(methodDefinition) {
            if (!methodDefinition.accessibility &&
                util.isTypeScriptFile(context.getFilename())) {
                context.report({
                    node: methodDefinition,
                    messageId: 'missingAccessibility',
                    data: {
                        type: 'method definition',
                        name: util.getNameFromClassMember(methodDefinition, sourceCode),
                    },
                });
            }
        }
        /**
         * Checks if property has an accessibility modifier.
         * @param classProperty The node representing a ClassProperty.
         */
        function checkPropertyAccessibilityModifier(classProperty) {
            if (!classProperty.accessibility &&
                util.isTypeScriptFile(context.getFilename())) {
                context.report({
                    node: classProperty,
                    messageId: 'missingAccessibility',
                    data: {
                        type: 'class property',
                        name: util.getNameFromPropertyName(classProperty.key),
                    },
                });
            }
        }
        return {
            ClassProperty: checkPropertyAccessibilityModifier,
            MethodDefinition: checkMethodAccessibilityModifier,
        };
    },
});
//# sourceMappingURL=explicit-member-accessibility.js.map