"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const util = __importStar(require("../util"));
exports.default = util.createRule({
    name: 'member-naming',
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Enforces naming conventions for class members by visibility.',
            category: 'Stylistic Issues',
            recommended: false,
        },
        messages: {
            incorrectName: '{{accessibility}} property {{name}} should match {{convention}}.',
        },
        schema: [
            {
                type: 'object',
                properties: {
                    public: {
                        type: 'string',
                        minLength: 1,
                        format: 'regex',
                    },
                    protected: {
                        type: 'string',
                        minLength: 1,
                        format: 'regex',
                    },
                    private: {
                        type: 'string',
                        minLength: 1,
                        format: 'regex',
                    },
                },
                additionalProperties: false,
                minProperties: 1,
            },
        ],
    },
    defaultOptions: [{}],
    create(context, [config]) {
        const sourceCode = context.getSourceCode();
        const conventions = Object.keys(config).reduce((acc, accessibility) => {
            acc[accessibility] = new RegExp(config[accessibility]);
            return acc;
        }, {});
        /**
         * Check that the property name matches the convention for its
         * accessibility.
         * @param {ASTNode} node the named node to evaluate.
         * @returns {void}
         * @private
         */
        function validateName(node) {
            const name = util.getNameFromClassMember(node, sourceCode);
            const accessibility = node.accessibility || 'public';
            const convention = conventions[accessibility];
            const method = node;
            if (method.kind === 'constructor')
                return;
            if (!convention || convention.test(name))
                return;
            context.report({
                node: node.key,
                messageId: 'incorrectName',
                data: { accessibility, name, convention },
            });
        }
        return {
            MethodDefinition: validateName,
            ClassProperty: validateName,
        };
    },
});
//# sourceMappingURL=member-naming.js.map