"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toBeDisabled = toBeDisabled;
exports.toBeEnabled = toBeEnabled;

var _jestMatcherUtils = require("jest-matcher-utils");

var _utils = require("./utils");

// form elements that support 'disabled'
var FORM_TAGS = ['fieldset', 'input', 'select', 'optgroup', 'option', 'button', 'textarea'];

function getTag(element) {
  return element.tagName && element.tagName.toLowerCase();
}
/*
 * According to specification:
 * If <fieldset> is disabled, the form controls that are its descendants,
 * except descendants of its first optional <legend> element, are disabled
 *
 * https://html.spec.whatwg.org/multipage/form-elements.html#concept-fieldset-disabled
 *
 * This method tests whether element is first legend child of fieldset parent
 */


function isFirstLegendChildOfFieldset(element, parent) {
  return getTag(element) === 'legend' && getTag(parent) === 'fieldset' && element.isSameNode(Array.from(parent.children).find(function (child) {
    return getTag(child) === 'legend';
  }));
}

function isElementDisabledByParent(element, parent) {
  return isElementDisabled(parent) && !isFirstLegendChildOfFieldset(element, parent);
}

function isElementDisabled(element) {
  return FORM_TAGS.includes(getTag(element)) && element.hasAttribute('disabled');
}

function isAncestorDisabled(element) {
  var parent = element.parentElement;
  return Boolean(parent) && (isElementDisabledByParent(element, parent) || isAncestorDisabled(parent));
}

function toBeDisabled(element) {
  var _this = this;

  (0, _utils.checkHtmlElement)(element, toBeDisabled, this);
  var isDisabled = isElementDisabled(element) || isAncestorDisabled(element);
  return {
    pass: isDisabled,
    message: function message() {
      var is = isDisabled ? 'is' : 'is not';
      return [(0, _jestMatcherUtils.matcherHint)(`${_this.isNot ? '.not' : ''}.toBeDisabled`, 'element', ''), '', `Received element ${is} disabled:`, `  ${(0, _jestMatcherUtils.printReceived)(element.cloneNode(false))}`].join('\n');
    }
  };
}

function toBeEnabled(element) {
  var _this2 = this;

  (0, _utils.checkHtmlElement)(element, toBeEnabled, this);
  var isEnabled = !(isElementDisabled(element) || isAncestorDisabled(element));
  return {
    pass: isEnabled,
    message: function message() {
      var is = isEnabled ? 'is' : 'is not';
      return [(0, _jestMatcherUtils.matcherHint)(`${_this2.isNot ? '.not' : ''}.toBeEnabled`, 'element', ''), '', `Received element ${is} enabled:`, `  ${(0, _jestMatcherUtils.printReceived)(element.cloneNode(false))}`].join('\n');
    }
  };
}