"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toHaveFormValues = toHaveFormValues;

var _jestMatcherUtils = require("jest-matcher-utils");

var _jestDiff = _interopRequireDefault(require("jest-diff"));

var _isEqual = _interopRequireDefault(require("lodash/isEqual"));

var _isEqualWith = _interopRequireDefault(require("lodash/isEqualWith"));

var _uniq = _interopRequireDefault(require("lodash/uniq"));

var _utils = require("./utils");

var _css = _interopRequireDefault(require("css.escape"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function compareArraysAsSet(a, b) {
  if (Array.isArray(a) && Array.isArray(b)) {
    return (0, _isEqual.default)(new Set(a), new Set(b));
  }

  return undefined;
}

function getSelectValue(_ref) {
  var multiple = _ref.multiple,
      selectedOptions = _ref.selectedOptions;

  if (multiple) {
    return [...selectedOptions].map(function (opt) {
      return opt.value;
    });
  }
  /* istanbul ignore if */


  if (selectedOptions.length === 0) {
    return undefined; // Couldn't make this happen, but just in case
  }

  return selectedOptions[0].value;
}

function getInputValue(inputElement) {
  switch (inputElement.type) {
    case 'number':
      return inputElement.value === '' ? null : Number(inputElement.value);

    case 'checkbox':
      return inputElement.checked;

    default:
      return inputElement.value;
  }
}

function getSingleElementValue(element) {
  /* istanbul ignore if */
  if (!element) {
    return undefined;
  }

  switch (element.tagName.toLowerCase()) {
    case 'input':
      return getInputValue(element);

    case 'select':
      return getSelectValue(element);

    default:
      return element.value;
  }
} // Returns the combined value of several elements that have the same name
// e.g. radio buttons or groups of checkboxes


function getMultiElementValue(elements) {
  var types = (0, _uniq.default)(elements.map(function (element) {
    return element.type;
  }));

  if (types.length !== 1) {
    throw new Error('Multiple form elements with the same name must be of the same type');
  }

  switch (types[0]) {
    case 'radio':
      {
        var theChosenOne = elements.find(function (radio) {
          return radio.checked;
        });
        return theChosenOne ? theChosenOne.value : undefined;
      }

    case 'checkbox':
      return elements.filter(function (checkbox) {
        return checkbox.checked;
      }).map(function (checkbox) {
        return checkbox.value;
      });

    default:
      // NOTE: Not even sure this is a valid use case, but just in case...
      return elements.map(function (element) {
        return element.value;
      });
  }
}

function getFormValue(container, name) {
  var elements = [...container.querySelectorAll(`[name="${(0, _css.default)(name)}"]`)];
  /* istanbul ignore if */

  if (elements.length === 0) {
    return undefined; // shouldn't happen, but just in case
  }

  switch (elements.length) {
    case 1:
      return getSingleElementValue(elements[0]);

    default:
      return getMultiElementValue(elements);
  }
} // Strips the `[]` suffix off a form value name


function getPureName(name) {
  return /\[\]$/.test(name) ? name.slice(0, -2) : name;
}

function getAllFormValues(container) {
  var names = Array.from(container.elements).map(function (element) {
    return element.name;
  });
  return names.reduce(function (obj, name) {
    return _extends({}, obj, {
      [getPureName(name)]: getFormValue(container, name)
    });
  }, {});
}

function toHaveFormValues(formElement, expectedValues) {
  var _this = this;

  (0, _utils.checkHtmlElement)(formElement, toHaveFormValues, this);

  if (!formElement.elements) {
    // TODO: Change condition to use instanceof against the appropriate element classes instead
    throw new Error('toHaveFormValues must be called on a form or a fieldset');
  }

  var formValues = getAllFormValues(formElement);
  return {
    pass: Object.entries(expectedValues).every(function (_ref2) {
      var name = _ref2[0],
          expectedValue = _ref2[1];
      return (0, _isEqualWith.default)(formValues[name], expectedValue, compareArraysAsSet);
    }),
    message: function message() {
      var to = _this.isNot ? 'not to' : 'to';
      var matcher = `${_this.isNot ? '.not' : ''}.toHaveFormValues`;
      var commonKeyValues = Object.keys(formValues).filter(function (key) {
        return expectedValues.hasOwnProperty(key);
      }).reduce(function (obj, key) {
        return _extends({}, obj, {
          [key]: formValues[key]
        });
      }, {});
      return [(0, _jestMatcherUtils.matcherHint)(matcher, 'element', ''), `Expected the element ${to} have form values`, (0, _jestDiff.default)(expectedValues, commonKeyValues)].join('\n\n');
    }
  };
}