"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkHtmlElement = checkHtmlElement;
exports.checkValidCSS = checkValidCSS;
exports.deprecate = deprecate;
exports.getMessage = getMessage;
exports.matches = matches;
exports.normalize = normalize;
exports.HtmlElementTypeError = void 0;

var _redent = _interopRequireDefault(require("redent"));

var _jestMatcherUtils = require("jest-matcher-utils");

var _css = require("css");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class HtmlElementTypeError extends Error {
  constructor(received, matcherFn, context) {
    super();
    /* istanbul ignore next */

    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, matcherFn);
    }

    var withType = '';

    try {
      withType = (0, _jestMatcherUtils.printWithType)('Received', received, _jestMatcherUtils.printReceived);
    } catch (e) {// Can throw for Document:
      // https://github.com/jsdom/jsdom/issues/2304
    }

    this.message = [(0, _jestMatcherUtils.matcherHint)(`${context.isNot ? '.not' : ''}.${matcherFn.name}`, 'received', ''), '', `${(0, _jestMatcherUtils.RECEIVED_COLOR)('received')} value must be an HTMLElement or an SVGElement.`, withType].join('\n');
  }

}

exports.HtmlElementTypeError = HtmlElementTypeError;

function checkHasWindow(htmlElement) {
  if (!htmlElement || !htmlElement.ownerDocument || !htmlElement.ownerDocument.defaultView) {
    for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      args[_key - 1] = arguments[_key];
    }

    throw new HtmlElementTypeError(htmlElement, ...args);
  }
}

function checkHtmlElement(htmlElement) {
  for (var _len2 = arguments.length, args = new Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
    args[_key2 - 1] = arguments[_key2];
  }

  checkHasWindow(htmlElement, ...args);
  var window = htmlElement.ownerDocument.defaultView;

  if (!(htmlElement instanceof window.HTMLElement) && !(htmlElement instanceof window.SVGElement)) {
    throw new HtmlElementTypeError(htmlElement, ...args);
  }
}

class InvalidCSSError extends Error {
  constructor(received, matcherFn) {
    super();
    /* istanbul ignore next */

    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, matcherFn);
    }

    this.message = [received.message, '', (0, _jestMatcherUtils.RECEIVED_COLOR)(`Failing css:`), (0, _jestMatcherUtils.RECEIVED_COLOR)(`${received.css}`)].join('\n');
  }

}

function checkValidCSS(css) {
  var ast = (0, _css.parse)(`selector { ${css} }`, {
    silent: true
  }).stylesheet;

  if (ast.parsingErrors && ast.parsingErrors.length > 0) {
    var _ast$parsingErrors$ = ast.parsingErrors[0],
        reason = _ast$parsingErrors$.reason,
        line = _ast$parsingErrors$.line;

    for (var _len3 = arguments.length, args = new Array(_len3 > 1 ? _len3 - 1 : 0), _key3 = 1; _key3 < _len3; _key3++) {
      args[_key3 - 1] = arguments[_key3];
    }

    throw new InvalidCSSError({
      css,
      message: `Syntax error parsing expected css: ${reason} on line: ${line}`
    }, ...args);
  }
}

function display(value) {
  return typeof value === 'string' ? value : (0, _jestMatcherUtils.stringify)(value);
}

function getMessage(matcher, expectedLabel, expectedValue, receivedLabel, receivedValue) {
  return [`${matcher}\n`, `${expectedLabel}:\n${(0, _jestMatcherUtils.EXPECTED_COLOR)((0, _redent.default)(display(expectedValue), 2))}`, `${receivedLabel}:\n${(0, _jestMatcherUtils.RECEIVED_COLOR)((0, _redent.default)(display(receivedValue), 2))}`].join('\n');
}

function matches(textToMatch, matcher) {
  if (matcher instanceof RegExp) {
    return matcher.test(textToMatch);
  } else {
    return textToMatch.includes(String(matcher));
  }
}

function deprecate(name, replacementText) {
  // Notify user that they are using deprecated functionality.
  // eslint-disable-next-line no-console
  console.warn(`Warning: ${name} has been deprecated and will be removed in future updates.`, replacementText);
}

function normalize(text) {
  return text.replace(/\s+/g, ' ').trim();
}