"use strict";

const _ = require("lodash");
const isStandardSyntaxAtRule = require("../../utils/isStandardSyntaxAtRule");
const report = require("../../utils/report");
const ruleMessages = require("../../utils/ruleMessages");
const validateOptions = require("../../utils/validateOptions");

const ruleName = "at-rule-property-requirelist";

const messages = ruleMessages(ruleName, {
  expected: (property, atRule) =>
    `Expected property "${property}" for at-rule "${atRule}"`
});

const rule = function(primary) {
  return (root, result) => {
    const validOptions = validateOptions(result, ruleName, {
      actual: primary,
      possible: [_.isObject]
    });

    if (!validOptions) {
      return;
    }

    root.walkAtRules(atRule => {
      if (!isStandardSyntaxAtRule(atRule)) {
        return;
      }

      const { name, nodes } = atRule;
      const atRuleName = name.toLowerCase();

      if (!primary[atRuleName]) {
        return;
      }

      primary[atRuleName].forEach(property => {
        const propertyName = property.toLowerCase();

        const hasProperty = nodes.find(
          node => node.prop.toLowerCase() === propertyName
        );

        if (hasProperty) {
          return;
        }

        return report({
          message: messages.expected(propertyName, atRuleName),
          node: atRule,
          result,
          ruleName
        });
      });
    });
  };
};

rule.ruleName = ruleName;
rule.messages = messages;

module.exports = rule;
