/* Norton's rendition of the MVC User Mgmt Page (June 2019) */
import { UserStore } from "./userStore.js";
import { UserDetailsView } from "./views/userDetailsView.js";
import { UserMasterView } from "./views/userMasterView.js";
import { UserManagementView } from "./views/userManagementView.js";
import { UserSimpleView } from "./views/userSimpleView.js";


class UserManagementController {

    constructor(router) {
        this.newUserInProcess = false;
        this.userStore = new UserStore();
        this.userManagementView = new UserManagementView(this);
        this.userDetailsView = new UserDetailsView(this);
        this.userMasterView = new UserMasterView(this);
        this.userSimpleView = new UserSimpleView(this);

        this.router = router;
        this.router.add("/user-management.html", request => {
            
            let params = this.getParamsFromRequest(request);
            let userObj = this.userStore.get(params.userId);
            this.load(userObj);
        });

        this.router.add("/user.html", request => {

            let params = this.getParamsFromRequest(request);
            let userObj = this.userStore.get(params.userId);
            
            this.userSimpleView.render(() => {
                this.router.setRouteLinks();
            }, userObj);
        });
    }

}

UserManagementController.prototype.getParamsFromRequest = function(request) {

    let { parameters } = request;

    let paramObj = {};
    for (let el of parameters) {
        let [ name, value ] = el.split("=");
        paramObj[name] = value;
    }

    return paramObj; 

}


UserManagementController.prototype.load = function(userObj) {

    this.userManagementView.render(() => {

        this.userMasterView.resetUserRows(this.userStore.getAll());
        
        if (userObj) {
            this.userDetailsView.populateUserDetails(userObj);
            this.userDetailsView.showInputForm();
        } else {
            this.userDetailsView.clearSlate();
            this.userDetailsView.hideInputForm();
        }
        
        this.router.setRouteLinks();

    }, userObj);

}

UserManagementController.prototype.new = function() {

    // throw new Error("Some error occurred.");
    let newId = this.userStore.uniqueId();
    this.userDetailsView.setCurrentUserId(newId);
    this.userDetailsView.clearSlate();
    this.userDetailsView.showInputForm();
    this.newUserInProcess = true;

}

UserManagementController.prototype.save = function(e) {

    e.preventDefault();

    // Check validity on the form
    if (this.userDetailsView.validateForm()) {

        let newUser = {};
        newUser.id = this.userDetailsView.getCurrentUserId();
    
        if (confirm (`Are you sure you want to save changes for user ${newUser.id}?`)) {
    
            newUser = this.userDetailsView.getFormInputValues();

            if (this.newUserInProcess) {
                this.userStore.add(newUser);
            } else {
                this.userStore.update(newUser);
            }
            
            this.newUserInProcess = false;
            this.load();
    
            this.userDetailsView.hideInputForm();
        }
    }
}

UserManagementController.prototype.cancel = function() {
    
    // cancel the edit and reload the same user or clear the 
    // Users Details view if the user was new

    if (this.newUserInProcess) {
        this.userDetailsView.clearSlate();
        this.userDetailsView.hideInputForm();
    } else {
        this.edit(null, this.userDetailsView.getCurrentUserId());
    }

}

UserManagementController.prototype.edit = function(e, id) {

    this.newUserInProcess = false;
    
    let userId;
    if (e) { userId = e.currentTarget.id.slice(1); } else userId = id;

    let currentUser = this.userStore.get(userId);

    this.userDetailsView.populateUserDetails(currentUser);
    this.userDetailsView.showInputForm();
}

UserManagementController.prototype.edit = function(e, id) {

    this.newUserInProcess = false;
    
    let userId;
    if (e) { userId = e.currentTarget.id.slice(1); } else userId = id;

    let currentUser = this.userStore.get(userId);

    this.userDetailsView.populateUserDetails(currentUser);
    this.userDetailsView.showInputForm();
}

UserManagementController.prototype.delete = function(e, id) {
    
    let userId;
    if (e) { userId = e.currentTarget.id.slice(1); } else userId = id;
    let userName = this.userStore.get(userId).name;

    if (confirm("Are you sure you want to delete " + userName + "?")) {
        this.userStore.remove(userId);    
        this.load();
    }
}

export { UserManagementController };