// Norton's spin on the Virtual World project #4;
// utilizes object classes, prototypes, and callbacks....
// Now compliant with ES6 strictness and exports

const ROWS = 40;
const COLS = 60;

function worldItemFactory() { }

worldItemFactory.build = function(symbol,latitude,longitude) {
    
    var type;
    var ableToMoveOn;

    switch (symbol) {
            
        case "#":
            type = "wall";
            break;

        case "p":
            type = "plant";
            ableToMoveOn = ["empty"];
            break;

        case "a":
            type = "animal";
            ableToMoveOn = ["empty", "plant"];
            break;

        case " ":
            type = "empty";
            break;
    }

    return new WorldItem(symbol,latitude,longitude,type,ableToMoveOn);
}

function WorldItem(symbol,latitude,longitude,type,ableToMoveOn) { 
 
	this.type = type;
	this.symbol = symbol;
	this.latitude = latitude;
	this.longitude = longitude;
	this.ableToMoveOn = ableToMoveOn;    

}

WorldItem.prototype.act = function(world) {

    let alive = true;
    let isPlant = this.type == "plant";
    let isAnimal = this.type == "animal";

    if (isPlant || isAnimal) {

        alive = Math.random() < 0.64 ? true : false;

        if (alive) {

            let shouldReproduce = Math.random() < 0.63 ? true : false;
            if (shouldReproduce) {
                this.reproduce(world);
            }

            if (isAnimal) {
                let shouldMove = Math.random() < 0.05 ? true : false;
                if (shouldMove) {
                    this.move(world);
                }
            }

        } else {

            world.remove(this);

        }
    }
}


WorldItem.prototype.notSelf = function(row,col) {
    return (!(row == this.latitude && col == this.longitude));
}


WorldItem.prototype.doIfInBoundsAndNotSelf = function(world, callback) {
    
    let y = this.latitude;
    let x = this.longitude;
    let radius = 1;

    let possibleMoves = [];

    // Create an array of possible moves:
    for (let r = y - radius; r <= y + radius; r++) {
        if (r >= 0 && r < world.getHeight()) {
            for (let c = x + radius; c >= x - radius; c--) {
                if (c >= 0 && c < world.getWidth() && this.notSelf(r,c)) {
                    possibleMoves.push({r: r, c: c});
                }
            }
        }
    }

    // Shuffle:
    possibleMoves = shuffle(possibleMoves);

    // Attempt the callback for each in turn; break if successful:
    for (let possibleMove of possibleMoves) {
        let target = world.getObjectAt(possibleMove.r,possibleMove.c);
        if (callback(target, this)) break;
    }
}

WorldItem.prototype.reproduce = function(world) {

    this.doIfInBoundsAndNotSelf(world, function(target, source) {
        
        if (target.type == "empty") {
            console.log(`Copying ${source.type} from ${source.latitude},${source.longitude} to ${target.latitude},${target.longitude}`);
            world.copy(target, source);
            return true;
        }

        return false;
    });
}

WorldItem.prototype.move = function(world) {
   
    this.doIfInBoundsAndNotSelf(world, function(target, source) {
        
        if (source.ableToMoveOn && source.ableToMoveOn.some(e => e == target.type)) {
            console.log(`Moving ${source.type} from ${source.latitude},${source.longitude} to ${target.latitude},${target.longitude}`);
            world.move(target, source);
            return true;
        }

        return false;
    });
}

class World {

    constructor(string) {
        this.grid = World.stringToGrid(string);
        this.rows = this.grid.length;
        this.cols = this.grid[0].length;
    }

}

// Static method not requiring instantiation of the object
World.stringToGrid = function(string) {

    let grid = [];
    let rowNum = 0;
    let colNum = 0;

    grid.push([]);

    for (let c of string) {
        if (c == '\n') {
            rowNum++;
            colNum = 0;
            grid.push([]);
            continue;
        }
        grid[rowNum].push(worldItemFactory.build(c, rowNum, colNum));
        colNum++;
    }

    return grid;

}

// Static method not requiring instantiation of the object
World.gridToString = function(grid) {

    let responseString = "";

    for (let i = 0; i < grid.length; i++) {
        for (let j = 0; j < grid[0].length; j++) {
            responseString += grid[i][j];
        }
        
        responseString += '\n';
    }
    return responseString.slice(0,responseString.length - 1);
    
}


World.prototype.toString = function() {

    let responseString = "";

    for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.cols; j++) {
            responseString += this.grid[i][j].symbol;
        }
        responseString += '\n';
    }

    return responseString.slice(0,responseString.length - 1);

}

World.prototype.move = function(targetWorldItem, sourceWorldItem) {

    this.copy(targetWorldItem, sourceWorldItem);
    this.remove(sourceWorldItem);
}

World.prototype.remove = function(worldItem) {
    // console.log(`Death: ${worldItem.type} @ ${worldItem.latitude},${worldItem.longitude}`);
    this.grid[worldItem.latitude][worldItem.longitude] = worldItemFactory.build(" ", worldItem.latitude, worldItem.longitude);
}

World.prototype.copy = function(targetWorldItem, sourceWorldItem) {
    this.grid[targetWorldItem.latitude][targetWorldItem.longitude] = worldItemFactory.build(sourceWorldItem.symbol, targetWorldItem.latitude, targetWorldItem.longitude);
}

World.prototype.getObjectAt = function(latitude, longitude) {
    return this.grid[latitude][longitude];
}

World.prototype.getHeight = function() {
    return this.grid.length;
}

World.prototype.getWidth = function() {
    return this.grid[0].length;
}

World.prototype.turn = function() {

    for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.cols; j++) {

            let current = this.getObjectAt(i,j);
            current.act(this);

        }
    }
}

// Borrowed from D3 code :-)
let shuffle = function(array, i0, i1) {
    if ((m = arguments.length) < 3) {
      i1 = array.length;
      if (m < 2) i0 = 0;
    }
    var m = i1 - i0, t, i;
    while (m) {
      i = Math.random() * m-- | 0;
      t = array[m + i0], array[m + i0] = array[i + i0], array[i + i0] = t;
    }
    return array;
};

export { World, WorldItem };
