// Norton's spin on the Virtual World project #2;
// utilizes object classes, prototypes, and callbacks....

const ROWS = 40;
const COLS = 80;

function worldItemFactory() { }

worldItemFactory.build = function(symbol,latitude,longitude) {
    
    var type;
    var ableToMoveOn;

    switch (symbol) {
            
        case "#":
            type = "wall";
            break;

        case "p":
            type = "plant";
            ableToMoveOn = ["empty"];
            break;

        case "a":
            type = "animal";
            ableToMoveOn = ["empty", "plant"];
            break;

        case " ":
            type = "empty";
            break;
    }

    return new WorldItem(symbol,latitude,longitude,type,ableToMoveOn);
}

function WorldItem(symbol,latitude,longitude,type,ableToMoveOn) { 
 
	this.type = type;
	this.symbol = symbol;
	this.latitude = latitude;
	this.longitude = longitude;
	this.ableToMoveOn = ableToMoveOn;    

}

WorldItem.prototype.act = function(world) {

    let alive = true;
    let isPlant = this.type == "plant";
    let isAnimal = this.type == "animal";

    if (isPlant || isAnimal) {

        alive = Math.random() < 0.68 ? true : false;

        if (alive) {

            shouldReproduce = Math.random() < 0.63 ? true : false;
            if (shouldReproduce) {
                this.reproduce(world);
            }

            if (isAnimal) {
                let shouldMove = Math.random() < 0.63 ? true : false;
                if (shouldMove) {
                    this.move(world);
                }
            }

        } else {

            world.remove(this);

        }
    }
}


WorldItem.prototype.notSelf = function(row,col) {
    return (!(row == this.latitude && col == this.longitude));
}


WorldItem.prototype.doIfInBoundsAndNotSelf = function(world, callback) {
    
    let y = this.latitude;
    let x = this.longitude;

    outer:
    for (let r = y - 1; r <= y + 1; r++) {

        if (r >= 0 && r < world.getHeight()) {

            for (let c = x+1; c >= x-1; c--) {

                if (c >= 0 && c < world.getWidth() && this.notSelf(r,c)) {

                    let target = world.getObjectAt(r,c);
                    if (callback(target, this)) break outer;

                }
            }
        }
    }
}

WorldItem.prototype.reproduce = function(world) {

    this.doIfInBoundsAndNotSelf(world, function(target, source) {
        
        if (target.type == "empty") {
            console.log(`Copying ${source.type} from ${source.latitude},${source.longitude} to ${target.latitude},${target.longitude}`);
            world.copy(target, source);
            return true;
        }

        return false;
    });
}

WorldItem.prototype.move = function(world) {
   
    this.doIfInBoundsAndNotSelf(world, function(target, source) {
        
        if (this.ableToMoveOn && this.ableToMoveOn.some(e => e == target.type)) {
            console.log(`Moving ${source.type} from ${source.latitude},${source.longitude} to ${target.latitude},${target.longitude}`);
            world.move(target, source);
            return true;
        }

        return false;
    });
}

class World {

    constructor(string) {
        this.grid = World.stringToGrid(string);
        this.rows = this.grid.length;
        this.cols = this.grid[0].length;
    }

}

// Static method not requiring instantiation of the object
World.stringToGrid = function(string) {

    let grid = [];
    let rowNum = 0;
    let colNum = 0;

    grid.push([]);

    for (let c of string) {
        if (c == '\n') {
            rowNum++;
            colNum = 0;
            grid.push([]);
            continue;
        }
        grid[rowNum].push(worldItemFactory.build(c, rowNum, colNum));
        colNum++;
    }

    return grid;

}

// Static method not requiring instantiation of the object
World.gridToString = function(grid) {

    responseString = "";

    for (let i = 0; i < grid.length; i++) {
        for (let j = 0; j < grid[0].length; j++) {
            responseString += grid[i][j];
        }
        
        responseString += '\n';
    }
    return responseString.slice(0,responseString.length - 1);
    
}


World.prototype.toString = function() {

    responseString = "";

    for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.cols; j++) {
            responseString += this.grid[i][j].symbol;
        }
        responseString += '\n';
    }

    return responseString.slice(0,responseString.length - 1);

}

World.prototype.move = function(targetWorldItem, sourceWorldItem) {

    this.copy(targetWorldItem, sourceWorldItem);
    this.remove(sourceWorldItem);
}

World.prototype.remove = function(worldItem) {
    console.log(`Death: ${worldItem.type} @ ${worldItem.latitude},${worldItem.longitude}`);
    this.grid[worldItem.latitude][worldItem.longitude] = worldItemFactory.build(" ", worldItem.latitude, worldItem.longitude);
}

World.prototype.copy = function(targetWorldItem, sourceWorldItem) {
    this.grid[targetWorldItem.latitude][targetWorldItem.longitude] = worldItemFactory.build(sourceWorldItem.symbol, targetWorldItem.latitude, targetWorldItem.longitude);
}

World.prototype.getObjectAt = function(latitude, longitude) {
    return this.grid[latitude][longitude];
}

World.prototype.getHeight = function() {
    return this.grid.length;
}

World.prototype.getWidth = function() {
    return this.grid[0].length;
}

World.prototype.turn = function() {

    for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.cols; j++) {

            let current = this.getObjectAt(i,j);
            current.act(this);

        }
    }
}



let myGrid = [];

// Initialize a world grid based on ROWS and COLS above
for (let i = 0; i < ROWS; i++) {
    
    myGrid[i] = [];
    for (let j = 0; j < COLS; j++) {

        if (i == 0 || j == 0 || i == ROWS-1 || j == COLS -1 ) {
            myGrid[i][j] = "#";
        } else {
            let next = ' ';
            let num = Math.random() * 10;
            if (num <= 2.5) {
                next = "#";
            } else if (num <= 5) {
                next = "p";
            } else if (num <= 7.5) {
                next = "a";
            } else { 
                next = " ";
            }

            myGrid[i][j] = next;

        }
    }
}

// Convert the above grid to a string
let gridString = World.gridToString(myGrid);
var world = new World(gridString);

animateWorld(world);