
import { coreMath } from "../lib/core/core-math.js";
import { coreObject } from "../lib/core/core-object.js";

function WorldView(worldItemTypeOptions) {

	this.tableElement;
	this.tableCellsIconElements = [];
	this.tableCellSize = 20;

	this.worldItemSymbolTypes = {};
	this.worldItemSymbolColorIcons = {};

	coreObject.forEach(worldItemTypeOptions, function(type, option) {

		this.worldItemSymbolTypes[option.symbol] = type;
		this.worldItemSymbolColorIcons[option.symbol] = option;

	}.bind(this));
}

WorldView.prototype.initialize = function(worldString) {

	this.createTable(worldString);
}

WorldView.prototype.render = function(worldString) {

	this.updateIconElements(worldString);
}

WorldView.prototype.createTable = function(worldString) {

	this.tableElement = document.createElement("table");
	var rowElement = document.createElement("tr");
	var tableCellCount = 0;

	this.forEachWorldItemSymbol(worldString, function(symbol) {

		if (symbol !== null) {

			var tableCellElement = document.createElement("td");
			tableCellElement.style.height = this.tableCellSize + "px";
			tableCellElement.style.lineHeight = this.tableCellSize + "px";
			tableCellElement.style.width = this.tableCellSize + "px";
			tableCellElement.style.minWidth = this.tableCellSize + "px";

			var iconElement = document.createElement("i");
			tableCellElement.appendChild(iconElement);

			this.updateTableElement(symbol, tableCellElement, iconElement);

			this.tableCellsIconElements.push({
				tableCellElement: tableCellElement,
				iconElement: iconElement
			});

			rowElement.appendChild(tableCellElement);

		} else {

			this.tableElement.appendChild(rowElement);
			tableCellCount++;

			rowElement = document.createElement("tr");
		}
	}.bind(this));

	this.tableElement.appendChild(rowElement);
	tableCellCount++;

	document.body.appendChild(this.tableElement);

	this.tableElement.style.height = (tableCellCount * this.tableCellSize) + "px";
	this.tableElement.style.width = (tableCellCount * this.tableCellSize) + "px";
}

WorldView.prototype.updateIconElements = function(worldString) {

	document.body.style.setProperty("display", "none");

	var index = 0;

	this.forEachWorldItemSymbol(worldString, function(symbol) {

		if (symbol != null) {

			var tableCellsIconElement = this.tableCellsIconElements[index];

			this.updateTableElement(symbol, tableCellsIconElement.tableCellElement, tableCellsIconElement.iconElement);

			index++;
		}

	}.bind(this));

	document.body.style.setProperty("display", "block");
}

WorldView.prototype.updateTableElement = function(symbol, tableCellElement, iconElement) {

	var worldItemType = this.worldItemSymbolTypes[symbol];
	var worldItemColor = this.worldItemSymbolColorIcons[symbol].color;
	var worldItemIcon = this.worldItemSymbolColorIcons[symbol].icon;

	if (worldItemType === "wall") {

		tableCellElement.style.backgroundColor = worldItemColor;

	} else if (worldItemType === "empty") {

		tableCellElement.style.backgroundColor = worldItemColor;
		iconElement.className = "";

	} else {

		iconElement.className = "fa fa-" + worldItemIcon;
		iconElement.style.color = worldItemColor;
		iconElement.style.transform = "";

		if (coreMath.randomBoolean()) {
			
			iconElement.style.transform = "scaleX(-1)";
		}
	}
}

WorldView.prototype.forEachWorldItemSymbol = function(worldString, callback) {

	worldString = worldString.trim();

	for (var index = 0; index < worldString.length; index++) {

		var symbol = worldString[index];

		if (symbol === "\n") {

			callback(null);

		} else {

			callback(symbol);
		}
	}
}


export { WorldView };