/* Norton's spin on the Virtual World project #7 (May 2019)

Contains its own animator and view layer now, the latter
based on the DOM and CSS style for elements.

Now compliant with ES6 strictness and import/export.
Follows the factory pattern; assumes an options object is 
passed in from the client to the worldPlanBuilder and 
WorldItemFactory object.

Refer to the index.html to see how the app.js is used as an ES6 module.
*/

function worldPlanBuilder(ROWS, COLS, itemOptions) {

    let symbolGrid = [];

    // Initialize a world grid based on ROWS and COLS above
    for (let i = 0; i < ROWS; i++) {
        
        symbolGrid[i] = [];
        for (let j = 0; j < COLS; j++) {

            // if (i == 0 || j == 0 || i == ROWS - 1 || j == COLS - 1 ) {

            //     symbolGrid[i][j] = itemOptions.wall.symbol;
            
            // } else {

                let next = itemOptions.empty.symbol;
                let num = Math.random();
                let refNum = 0;

                for (let i in itemOptions) {

                    refNum += itemOptions[i].buildChance;
                    if (num <= refNum) {
                        next = itemOptions[i].symbol;
                        break;
                    }

                }

                symbolGrid[i][j] = next;

            // }
        }
    }

    // Convert the above grid to a string (worldPlan)
    let worldPlan = symbolGridToString(symbolGrid);
    return worldPlan;
}

function symbolGridToString(symbolGrid) {

    let responseString = "";

    for (let i = 0; i < symbolGrid.length; i++) {
        for (let j = 0; j < symbolGrid[0].length; j++) {
            responseString += symbolGrid[i][j];
        }
        
        responseString += '\n';
    }
    return responseString.slice(0,responseString.length - 1);
    
}


function zeroPad(number, width) {
    let string = String(number);
    while (string.length < width) {
      string = "0" + string;
    }
    return string;
}


class World {

    constructor(worldPlan, worldItemFactory) {

        this.worldItemFactory = worldItemFactory;
        this.grid = this.stringToGrid(worldPlan);
        this.rows = this.grid.length;
        this.cols = this.grid[0].length;
    }

}

World.prototype.stringToGrid = function(string) {

    let grid = [];
    let rowNum = 0;
    let colNum = 0;

    grid.push([]);

    for (let c of string) {
        if (c == '\n') {
            rowNum++;
            colNum = 0;
            grid.push([]);
            continue;
        }
        grid[rowNum].push(this.worldItemFactory.buildFromSymbol(c, rowNum, colNum));
        colNum++;
    }

    return grid;

}

World.prototype.viewGrid = function() {

    // Iterate through the grid; return similar item with only the properties
    // needed by the View

    let viewGrid = [];

    for (let i = 0; i < this.rows; i++) {
        viewGrid.push([]);
        for (let j = 0; j < this.cols; j++) {
            
            let item = this.grid[i][j];
            let viewGridItem = {};

            for (let p in item) {
                // if (item[p] !== undefined && Object.getPrototypeOf(item[p]) !== Function.prototype) {
                
                if (p == 'latitude' || p == 'longitude' || p == 'health' || p == 'symbol' || p == 'type') {
                    viewGridItem[p] = item[p];
                }
            }

            viewGrid[i][j] = viewGridItem;
        }
    }

    return viewGrid;
}

World.prototype.toString = function() {

    let responseString = "";

    for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.cols; j++) {
            
            let item = this.grid[i][j];
            responseString += item.symbol;
        }
        responseString += '\n';
    }

    return responseString.slice(0,responseString.length - 1);

}

World.prototype.move = function(targetWorldItem, sourceWorldItem) {

    this.copy(targetWorldItem, sourceWorldItem);
    this.remove(sourceWorldItem);
}

World.prototype.eat = function(targetWorldItem, sourceWorldItem) {

    targetWorldItem.health -= 3;
    sourceWorldItem.health += 3;    
}


World.prototype.remove = function(worldItem) {
    // console.log(`Death: ${worldItem.type} @ ${worldItem.latitude},${worldItem.longitude}`);
    this.grid[worldItem.latitude][worldItem.longitude] = this.worldItemFactory.buildFromSymbol("z", worldItem.latitude, worldItem.longitude);
}

World.prototype.copy = function(targetWorldItem, sourceWorldItem) {

    let newType;

    if (sourceWorldItem.type === 'carnivore') {
        if (Math.random() < sourceWorldItem.vegetarianChance) {
            newType = 'animal';
        } else newType = sourceWorldItem.type;
    } else newType = sourceWorldItem.type;

    let newItem = this.worldItemFactory.buildFromType(newType, targetWorldItem.latitude, targetWorldItem.longitude);
    this.grid[targetWorldItem.latitude][targetWorldItem.longitude] = newItem;
}

World.prototype.plant = function(targetWorldItem) {

    let newItem = this.worldItemFactory.buildFromType("plant", targetWorldItem.latitude, targetWorldItem.longitude);
    this.grid[targetWorldItem.latitude][targetWorldItem.longitude] = newItem;
}

World.prototype.getObjectAt = function(latitude, longitude) {
    return this.grid[latitude][longitude];
}

World.prototype.getHeight = function() {
    return this.grid.length;
}

World.prototype.getWidth = function() {
    return this.grid[0].length;
}

World.prototype.turn = function() {

    for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.cols; j++) {

            let current = this.getObjectAt(i,j);
            
            if (current.type != "wall" && current.type != "empty") {
                current.act(this);
            }

        }
    }
}

export { World, worldPlanBuilder };