/* Norton's VirtualLife rendition (2019) */
import { StatsView } from './statsView.js'
const scale = 20;

function zeroPad(number, width) {
  let string = String(number);
  while (string.length < width) {
    string = "0" + string;
  }
  return string;
}

// Haverbeke's elt function
function elt(name, attrs, ...children) {
  let dom = document.createElement(name);

  for (let attr of Object.keys(attrs)) {
    dom.setAttribute(attr, attrs[attr]);
  }

  for (let child of children) {
    dom.appendChild(child);
  }

  return dom;
}

function drawLifeGrid(worldGrid) {
  return elt(
    "table",
    {
      id: "virtualWorld",
      class: "background",
      style: `width: ${worldGrid[0].length * (scale + 2)}px`
    },
    ...worldGrid.map((row, y) =>
      elt(
        "tr",
        {
          style: `height: ${scale}px`
        },
        ...row.map((item, x) => {
          let popupText = document.createElement("span");
          popupText.innerText="";
          popupText.className="popupText";

          let spriteElt = elt("td", {
            class: `${item.symbol}`,
            id: `${zeroPad(y, 3)}y${zeroPad(x, 3)}x`
          }, popupText);

          spriteElt.addEventListener('mouseover', toggleShowStats);
          spriteElt.addEventListener('mouseout', toggleShowStats);

          return spriteElt;

        })
      )
    )
  );
}

function toggleShowStats(e) {
  let cell = e.currentTarget;
  cell.firstChild.classList.toggle("showStats");
}

function WorldView(worldGrid, parent, worldAnimator) {

  this.worldGrid = [];
  this.worldAnimator = worldAnimator;
  this.statsView = new StatsView(this.worldAnimator);

  for (let inner of worldGrid) {
    this.worldGrid.push(inner.slice(0));
  }

  this.dom = elt("div", { class: "life" }, drawLifeGrid(this.worldGrid));
  this.initialized = false;
  this.parent = parent;
}

WorldView.prototype.clear = function() {
  this.dom.remove();
};

WorldView.prototype.initialize = function() {
  
  this.clear();
  this.dom.className = `${status}`;
  this.parent.appendChild(this.statsView.dom);
  this.parent.appendChild(this.dom);
  this.initialized = true;

};

WorldView.prototype.render = function(newWorldGrid, worldTurnTime) {

  let start = new Date();

  for (let y = 0; y < newWorldGrid.length; y++) {
    for (let x = 0; x < newWorldGrid[0].length; x++) {
      if (
        newWorldGrid[y][x].symbol != this.worldGrid[y][x].symbol ||
        (newWorldGrid[y][x].health &&
          newWorldGrid[y][x].health != this.worldGrid[y][x].health)
      ) {
        this.updateGrid(y, x, newWorldGrid[y][x]);
        this.worldGrid[y][x] = newWorldGrid[y][x];
      }
    }
  }

  let stop = new Date();

  let worldRenderTime = stop - start;
  this.statsView.render(newWorldGrid, worldRenderTime, worldTurnTime);

};

WorldView.prototype.updateGrid = function(y, x, newItem) {
  let thisElement = document.getElementById(
    `${zeroPad(y, 3)}y${zeroPad(x, 3)}x`
  );

  thisElement.className = `${newItem.symbol}`;

  if (Math.random() < .2) thisElement.firstChild.innerText = `Health: ${Math.floor(newItem.health)}`;

  if (newItem.health > 75) {
    thisElement.classList.add("vibrant");
  } else if (newItem.health < 25) {
    thisElement.classList.add("sickly");
  }

};

export { WorldView };
